// Class to minimize a DFA

package fsa_editor;

import au.edu.monash.csse.tonyj.cider.interpreter.*;
import au.edu.monash.csse.tonyj.cider.canvas.*;
import java.awt.Color;
import java.util.Set;
import java.util.TreeSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;
import java.util.LinkedList;
import java.util.NoSuchElementException;

public class DFAMinimizer implements GrammarSymbolListener {

	// Private variables
	private Set		stateSet;
	private Set		transitionSet;
	private Set		startArcSet;
	private Set		statePairSet;
	private Vector		alphabet;
	private Set		tempTransitionSet;

	private InterpretedTokenCanvas	canvas;

	private Color		uncheckedColour;
	private Color		checkedColour;

	// Needed for stages of minimization
	private int			stage;
	private static final int	CHECK_PAIRINGS_STAGE = 1;
	private static final int	MERGE_PAIRED_STATES_STAGE = 2;

	// Constructor
	public DFAMinimizer(InterpretedTokenCanvas c, int tag)
	{
		canvas = c;
		canvas.getInterpreter().addGrammarSymbolListener(this);

		stateSet = new HashSet();
		transitionSet = new HashSet();
		startArcSet = new HashSet();
		statePairSet = new HashSet();
		alphabet = new Vector();
		tempTransitionSet = new HashSet();

		uncheckedColour = new Color(220, 220, 220);
		checkedColour = Color.red;

		initialize(tag);
		createStatePairs();

		stage = CHECK_PAIRINGS_STAGE;
	}

	// Finalize method
	public void finalize() throws Throwable
	{
		canvas.getInterpreter().removeGrammarSymbolListener(this);
	}

	// Get all transition and state symbols from the parse forest for specified tag number
	// Determines the FSA alphabet from the transitions
	// (will treat regular expressions as unique alphabet members)
	private void initialize(int tag)
	{
		Iterator iter = canvas.getInterpreter().getAllSymbols().iterator();
		while (iter.hasNext()) {
			Object obj = iter.next();
			if (obj instanceof State && ((State) obj).get_tag() == tag)
				stateSet.add(obj);
			else if (obj instanceof Transition && ((Transition) obj).get_tag() == tag)
				transitionSet.add(obj);
			else if (obj instanceof StartArc && ((StartArc) obj).get_state().get_tag() == tag)
				startArcSet.add(obj);
		}
	
		// Calculate the alphabet
		TreeSet alphabetSet = new TreeSet();
		iter = transitionSet.iterator();
		while (iter.hasNext()) {
			Transition transition = (Transition) iter.next();
			alphabetSet.add(transition.get_transition());
		}
		alphabet.addAll(alphabetSet);
	}

	// Creates all the state pair objects
	// (only if both states in pair are Normal or both are Final)
	private void createStatePairs()
	{
		Object stateArray[] = stateSet.toArray();

		for (int i = 0; i < stateArray.length; i++) {
			State state1 = (State) stateArray[i];
			for (int j = i+1; j < stateArray.length; j++) {
				State state2 = (State) stateArray[j];
				if ((state1 instanceof NormalState && state2 instanceof NormalState) ||
				    (state1 instanceof FinalState && state2 instanceof FinalState))
					canvas.add(new DrawableLine(state1.get_mid_x(), state1.get_mid_y(), state2.get_mid_x(), state2.get_mid_y(),
						0.0, 0.0, false, uncheckedColour));
			}
		}
	}

	// Determines the state is linked to from the state passed to it on the specified transition
	// Returns null if none is found
	private State calculateToState(State from, String transitionStr)
	{
		State to = null;

		Iterator iter = transitionSet.iterator();
		while (iter.hasNext()) {
			Transition transition = (Transition) iter.next();
			if (transition.get_transition().equals(transitionStr) && from == transition.get_from()) {
					to = transition.get_to();
					break;
			}
		}
		return to;
	}

	// Returns state pair object for a pair of states passed to method, or null if none exists
	private StatePair getStatePair(State state1, State state2)
	{
		Iterator iter = statePairSet.iterator();
		while (iter.hasNext()) {
			StatePair pair = (StatePair) iter.next();
			if (pair.get_state1() == state1 && pair.get_state2() == state2)
				return pair;
			if (pair.get_state1() == state2 && pair.get_state2() == state1)
				return pair;
		}
		return null;
	}

	// Does the two stage processing step by step
	public boolean processNextStep()
	{
		if (stage == CHECK_PAIRINGS_STAGE) {
			if (!checkPairingsStep())
				stage = MERGE_PAIRED_STATES_STAGE;
			return true;
		}
		else 
			return mergePairedStates();
	}

	// Elimiates another state pair that is not valid
	// Returns false if there is nothing left to process, otherwise returns true
	private boolean checkPairingsStep()
	{
		// Go through each state pair
		Iterator iter = statePairSet.iterator();
		while (iter.hasNext()) {
			StatePair pair = (StatePair) iter.next();
			
			// Go through alphabet
			for (int i = 0; i < alphabet.size(); i++) {
				try {
					String transitionStr = (String) alphabet.elementAt(i);
					State dest1 = calculateToState(pair.get_state1(), transitionStr);
					State dest2 = calculateToState(pair.get_state2(), transitionStr);
					StatePair destinationPair = getStatePair(dest1, dest2);

					if (destinationPair == null && dest1 != dest2) {
						iter.remove();
						canvas.getInterpreter().removeSubTree(pair);
						return true;
					}
				}
				catch (Exception e) { ; }
			}
		}

		// If here, no eliminations were possible, so change the colours and return false
		iter = statePairSet.iterator();
		while (iter.hasNext()) {
			StatePair pair = (StatePair) iter.next();
			pair.get_line().set_colour(checkedColour);
		}
		return false;
	}

	// Merges a pair of states in the FSA
	private boolean mergePairedStates()
	{
		// See if there are any state pairs left to merge
		if (statePairSet.isEmpty())
			return false;

		Iterator iter = statePairSet.iterator();
		StatePair pair = (StatePair) iter.next();
		State state1 = pair.get_state1();
		State state2 = pair.get_state2();

		// If state 2 is a start state, then swap them
		Iterator iter2 = startArcSet.iterator();
		while (iter2.hasNext()) {
			StartArc sa = (StartArc) iter2.next();
			if (sa.get_state() == state2) {
				state1 = pair.get_state2();
				state2 = pair.get_state1();
				break;
			}
		}

		// Remove all transitions that link from state 2 to anywhere
		iter2 = transitionSet.iterator();
		while (iter2.hasNext()) {
			Transition transition = (Transition) iter2.next();
			if (transition.get_from() == state2) {
				iter2.remove();
				canvas.getInterpreter().removeSubTree(transition);
			}
		}

		// Move all transitions that link to state 2 to now link to state 1
		tempTransitionSet.clear();
		iter2 = transitionSet.iterator();
		while (iter2.hasNext()) {
			Transition transition = (Transition) iter2.next();
			if (transition.get_to() == state2) {

				// Remove old transition
				iter2.remove();
				canvas.getInterpreter().removeSubTree(transition);

				// Create new transition
				canvas.add(new DrawableArrow(transition.get_start_x(), transition.get_start_y(), transition.get_mid_x(),
					transition.get_mid_y(), state1.get_mid_x(), state1.get_mid_y(), 0.0, 0.0, false, null));
				canvas.add(new DrawableText(transition.get_transition(), transition.get_mid_x(), transition.get_mid_y(), 
					false, null));
			}
		}
		transitionSet.addAll(tempTransitionSet);

		// Remove state pair object and all other state pair objects that involve state 2
		iter.remove();
		canvas.getInterpreter().removeSubTree(pair);
		while (iter.hasNext()) {
			pair = (StatePair) iter.next();
			if (pair.get_state1() == state2 || pair.get_state2() == state2) {
				iter.remove();
				canvas.getInterpreter().removeSubTree(pair);
			}
		}

		// Change name of state 1
		iter2 = state1.getChildren().iterator();
		while (iter2.hasNext()) {
			Object obj = iter2.next();
			if (obj instanceof Text) {
				((Text) obj).set_label(state1.get_label() + "," + state2.get_label());
				break;
			}
		}

		// Remove state 2
		canvas.getInterpreter().removeSubTree(state2);
		return true;
	}

	// Needed for Grammar Symbol Listener
	public void symbolModified(GrammarSymbolEvent gse) { ; }
	public void symbolRemoved(GrammarSymbolEvent gse) { ; }

	public void symbolAdded(GrammarSymbolEvent gse)
	{
		if (gse.getGrammarSymbol() instanceof StatePair)
			statePairSet.add(gse.getGrammarSymbol());
		else if (gse.getGrammarSymbol() instanceof Transition)
			tempTransitionSet.add(gse.getGrammarSymbol());
	}
}

