// Simple example editor for drawing Finite State Automata

package fsa_editor;

import au.edu.monash.csse.tonyj.cider.interpreter.*;
import au.edu.monash.csse.tonyj.cider.canvas.InterpretedTokenCanvas;
import au.edu.monash.csse.tonyj.cider.canvas.DrawableToken;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import java.awt.event.KeyListener;
import java.awt.event.KeyEvent;
import java.awt.geom.Point2D;
import java.awt.geom.CubicCurve2D;
import javax.swing.JFrame;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import java.awt.Toolkit;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Color;
import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

public class Editor extends JFrame implements ActionListener, KeyListener, MouseListener, GrammarSymbolListener {

	// GUI elements
	private JButton			normalStateButton;
	private JButton			finalStateButton;
	private JButton			arcButton;
	private JButton			addArrow;
	private JButton			addCircle;
	private JButton			addText;
	private JButton			nextObject;
	private JButton			previousObject;
	private JButton			deleteObject;
	private JButton			printForest;
	private JButton			processButton;
	private JButton			createFSA;
	private JButton			NFAtoDFA;
	private JButton			minimizeDFA;
	private JButton			tagFSAs;
	private JButton			layout;
	private JButton			quit;
	private JPanel			buttonPanel;
	private JPanel			allPanel;
	private JTextArea		statusField;
	private JScrollPane		statusPane;
	private InterpretedTokenCanvas	canvas;

	// Needed for mouse dragging
	private boolean			mouseDrag;
	private int			startX;
	private int			startY;

	// Needed to deactivating solver constraints on a symbol
	private boolean			shiftKeyDown;

	// Needed for undo	
	private int			undoIndex;

	// Needed for conversion and minimization
	private NFAConverter		converter;
	private DFAMinimizer		minimizer;
	private	int			minimizerState;

	// Constants
	public static final int		IMAGE_WIDTH = 600;
	public static final int		IMAGE_HEIGHT = 400;
	public static final int		MOUSE_TOLERANCE = 5;
	public static final int		MIN_RADIUS = 10;

	// Constructor
	public Editor()
	{
		mouseDrag = false;
		shiftKeyDown = false;

		// Make buttons
		normalStateButton = new JButton("Add Normal State");
		finalStateButton = new JButton("Add Final State");
		arcButton = new JButton("Add Arc");
		addArrow = new JButton("Add Arrow");
		addCircle = new JButton("Add Circle");
		addText = new JButton("Add Text");
		nextObject = new JButton("Next Object");
		previousObject = new JButton("Previous Object");
		deleteObject = new JButton("Delete Object");
		printForest = new JButton("Print Parse Forest");
		processButton = new JButton("Process Input");
		createFSA = new JButton("Create FSA");
		NFAtoDFA = new JButton("Convert NFA to DFA");
		minimizeDFA = new JButton("Minimize DFA");
		tagFSAs = new JButton("Tag FSAs");
		layout = new JButton("Layout FSA");
		quit = new JButton("Quit");

		// Make button panel
		buttonPanel = new JPanel(new GridLayout(17, 1, 5, 5));
		buttonPanel.add(normalStateButton);
		buttonPanel.add(finalStateButton);
		buttonPanel.add(arcButton);
		buttonPanel.add(addArrow);
		buttonPanel.add(addCircle);
		buttonPanel.add(addText);
		buttonPanel.add(nextObject);
		buttonPanel.add(previousObject);
		buttonPanel.add(printForest);
		buttonPanel.add(deleteObject);
		buttonPanel.add(processButton);
		buttonPanel.add(createFSA);
		buttonPanel.add(NFAtoDFA);
		buttonPanel.add(minimizeDFA);
		buttonPanel.add(tagFSAs);
		buttonPanel.add(layout);
		buttonPanel.add(quit);

		// Make text field
		statusField = new JTextArea("No Current Object", 1, 60);
		statusField.setEditable(false);
		statusPane = new JScrollPane(statusField, ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER,
			ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);

		// Make token canvas
		canvas = new InterpretedTokenCanvas(IMAGE_WIDTH, IMAGE_HEIGHT, new FSAInterpreter());
		canvas.getInterpreter().addGrammarSymbolListener(this);
		canvas.getInterpreter().setLoggingLevel(java.util.logging.Level.FINE);

		// Add elements to the all panel
		allPanel = new JPanel(new BorderLayout(10, 10));
		allPanel.add(buttonPanel, BorderLayout.WEST);
		allPanel.add(canvas, BorderLayout.CENTER);
		allPanel.add(statusPane, BorderLayout.SOUTH);
		
		// Add all panel to frame and initialize 
		this.getContentPane().setLayout(new FlowLayout());
		this.getContentPane().add(allPanel);
		this.pack();

		// Add listeners
		normalStateButton.addActionListener(this);
		finalStateButton.addActionListener(this);
		arcButton.addActionListener(this);
		addArrow.addActionListener(this);
		addCircle.addActionListener(this);
		addText.addActionListener(this);
		nextObject.addActionListener(this);
		previousObject.addActionListener(this);
		deleteObject.addActionListener(this);
		processButton.addActionListener(this);
		createFSA.addActionListener(this);
		NFAtoDFA.addActionListener(this);
		minimizeDFA.addActionListener(this);
		tagFSAs.addActionListener(this);
		layout.addActionListener(this);
		printForest.addActionListener(this);
		quit.addActionListener(this);
		canvas.addKeyListener(this);
		normalStateButton.addKeyListener(this);
		finalStateButton.addKeyListener(this);
		arcButton.addKeyListener(this);
		addText.addKeyListener(this);
		addArrow.addKeyListener(this);
		addCircle.addKeyListener(this);
		nextObject.addKeyListener(this);
		previousObject.addKeyListener(this);
		deleteObject.addKeyListener(this);
		processButton.addKeyListener(this);
		createFSA.addKeyListener(this);
		NFAtoDFA.addKeyListener(this);
		minimizeDFA.addKeyListener(this);
		tagFSAs.addKeyListener(this);
		layout.addKeyListener(this);
		printForest.addKeyListener(this);
		quit.addKeyListener(this);
		buttonPanel.addKeyListener(this);
		allPanel.addKeyListener(this);
		canvas.addMouseListener(this);

		// Show the frame
		this.setTitle("FSA Editor");
		this.show();
	}

	// Updates the status field with details of the current object
	private void updateStatusField()
	{
		if (shiftKeyDown)
			statusField.setForeground(Color.red);
		else
			statusField.setForeground(Color.black);
		if (canvas.getNumTokens() == 0)
			statusField.setText("No Current Object");
		else
			statusField.setText("Current Object: " + ((GrammarSymbol) canvas.getTopToken()).toString());
	}

	// Deals with button presses
	public void actionPerformed(ActionEvent ae)
	{
		if (ae.getSource() == quit)
			System.exit(0);
		if (ae.getSource() == nextObject) {
			if (shiftKeyDown)
				return;
			if (canvas.getTopToken() == null)
				return;
			deactivateTop();
			canvas.moveToBottom(canvas.getTopToken());
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == previousObject) {
			if (shiftKeyDown)
				return;
			if (canvas.getTopToken() == null)
				return;
			deactivateTop();
			canvas.moveToTop(canvas.getBottomToken());
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == printForest) 
			canvas.getInterpreter().printForest(System.out);
		else if (ae.getSource() == deleteObject) {
			if (shiftKeyDown)
				return;
			if (canvas.getTopToken() == null)
				return;
			canvas.remove(canvas.getTopToken());
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == addCircle) {
			if (shiftKeyDown)
				return;
			deactivateTop();
			DrawableCircle dc = new DrawableCircle(20.0, 40.0, 40.0, true, null);
			canvas.add(dc);
			canvas.moveToTop(dc);
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == addArrow) {
			if (shiftKeyDown)
				return;
			deactivateTop();
			DrawableArrow da = new DrawableArrow(20.0, 40.0, 40.0, 40.0, 60.0, 40.0, 0.0, 0.0, true, null);
			canvas.add(da);
			canvas.moveToTop(da);
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == addText) {
			if (shiftKeyDown)
				return;
			deactivateTop();
			DrawableText dt = new DrawableText("", 40.0, 30.0, true, null);
			canvas.add(dt);
			canvas.moveToTop(dt);
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == normalStateButton) {
			if (shiftKeyDown)
				return;
			deactivateTop();
			canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_CreateNormalState);
			canvas.moveToTop(canvas.getBottomToken());
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == finalStateButton) {
			if (shiftKeyDown)
				return;
			deactivateTop();
			canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_CreateFinalState);
			canvas.moveToTop(canvas.getBottomToken());
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == arcButton) {
			if (shiftKeyDown)
				return;
			deactivateTop();
			canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_CreateArc);
			canvas.moveToTop(canvas.getBottomToken());
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == processButton) {
			deactivateTop();
			canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_MopUpEmpties);
			boolean success = canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_CheckForConclusion);
			if (!success)
				success = canvas.getInterpreter().applyTransformationInParallel(FSAInterpreter.TRANSFORMATION_ProcessInput);
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == createFSA) {
			deactivateTop();
			boolean success = canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_CreateFSA);
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == layout) {
			canvas.getInterpreter().applyForceDirectedLayout(1);
		}
		else if (ae.getSource() == NFAtoDFA) {
			deactivateTop();
			if (converter == null) {
				converter = new NFAConverter(canvas, 0);
				canvas.getInterpreter().applyForceDirectedLayout(1);
			}
			else {
				if (!converter.processNextStep()) {
					converter = null;
					canvas.add(new DrawableText("*** Conversion Complete ***", 200.0, 20.0, false, Color.magenta));
				}
				else 
					canvas.getInterpreter().applyForceDirectedLayout(1);
			}
			activateTop();
			updateStatusField();
			canvas.repaint();
		}
		else if (ae.getSource() == minimizeDFA) {
			/*
			if (minimizer == null)
				minimizer = new DFAMinimizer(canvas, 0);
			else {
				if (!minimizer.processNextStep()) {
					minimizer = null;
					canvas.add(new DrawableText("*** Minimization Complete ***", 200.0, 20.0, false, Color.magenta));
				}
			}
			canvas.repaint();
			*/
			if (minimizerState == 0) {
				canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_Min_CreateStatePairs);
				minimizerState = 1;
			}
			else if (minimizerState == 1) {
				canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_Min_ClearMarkedBits);
				if (!canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_Min_MarkInvalidStatePair)) {
					minimizerState = 2;
					canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_Min_ChangeColour);
				}
			}
			else {
				if (!canvas.getInterpreter().applyTransformation(FSAInterpreter.TRANSFORMATION_Min_MergeStatePair)) {
					minimizerState = 0;
					canvas.add(new DrawableText("*** Minimization Complete ***", 200.0, 20.0, false, Color.magenta));
				}
			}
			canvas.repaint();
		}
		// else if (ae.getSource() == tagFSAs) 
		//	tagFSAs();
	}

	// Activates the top token
	private void activateTop()
	{
		if (canvas.getNumTokens() != 0) {
			if (canvas.getTopToken() instanceof Text)
				((Text) canvas.getTopToken()).set_active(true);
			else if (canvas.getTopToken() instanceof Circle)
				((Circle) canvas.getTopToken()).set_active(true);
			else if (canvas.getTopToken() instanceof Arrow)
				((Arrow) canvas.getTopToken()).set_active(true);
		}
	}

	// Deactivates the top token
	private void deactivateTop()
	{
		if (canvas.getNumTokens() != 0) {
			if (canvas.getTopToken() instanceof Text)
				((Text) canvas.getTopToken()).set_active(false);
			else if (canvas.getTopToken() instanceof Circle)
				((Circle) canvas.getTopToken()).set_active(false);
			else if (canvas.getTopToken() instanceof Arrow)
				((Arrow) canvas.getTopToken()).set_active(false);
		}
	}

	// Deal with keyboard input
	public void keyPressed(KeyEvent ke) 
	{ 
		if (canvas.getNumTokens() != 0 && ke.getKeyCode() == KeyEvent.VK_SHIFT) {
			shiftKeyDown = true;
			updateStatusField();
		}
	}

	public void keyReleased(KeyEvent ke) 
	{ 
		if (canvas.getNumTokens() != 0 && ke.getKeyCode() == KeyEvent.VK_SHIFT) {
			shiftKeyDown = false;
			updateStatusField();
		}
	}

	public void keyTyped(KeyEvent ke)
	{
		if (canvas.getTopToken() instanceof Text) {
			Text t = (Text) canvas.getTopToken();
			if (ke.getKeyChar() == '\b') {
				if (!t.get_label().equals(""))
					t.set_label(t.get_label().substring(0, t.get_label().length() - 1));
				updateStatusField();
				canvas.repaint();
			}
			else if ((ke.getKeyChar() >= 'a' && ke.getKeyChar() <= 'z') || (ke.getKeyChar() >= 'A' && ke.getKeyChar() <= 'Z') ||
				(ke.getKeyChar() >= '0' && ke.getKeyChar() <= '9') || (ke.getKeyChar() == '_') || (ke.getKeyChar() == '*') ||
				(ke.getKeyChar() == '|') || (ke.getKeyChar() == '(') || (ke.getKeyChar() == ')')) {
				t.set_label(t.get_label() + ke.getKeyChar());
				updateStatusField();
				canvas.repaint();
			}
		}
	}

	// Deal with mouse input
	public void mouseEntered(MouseEvent me) { ; }
	public void mouseExited(MouseEvent me) { ; }
	public void mouseClicked(MouseEvent me) { ; }

	public void mousePressed(MouseEvent me)
	{
		if (me.getButton() != MouseEvent.BUTTON1 || canvas.getNumTokens() == 0)
			return;
		startX = me.getX();
		startY = me.getY();
		mouseDrag = true;
	}

	public void mouseReleased(MouseEvent me)
	{
		if (me.getButton() != MouseEvent.BUTTON1 || !mouseDrag)
			return;

		// Don't get movement outside canvas
		if (me.getX() < 0 || me.getX() > IMAGE_WIDTH || me.getY() < 0 || me.getY() > IMAGE_HEIGHT)
			return;

		GrammarSymbol current = (GrammarSymbol) canvas.getTopToken();
		if (current instanceof Text) {
			Text t = (Text) current;
			if (startX > ((int) t.get_mid_x() - MOUSE_TOLERANCE) && startX < ((int) t.get_mid_x() + MOUSE_TOLERANCE) &&
				startY > ((int) t.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) t.get_mid_y() + MOUSE_TOLERANCE)) {
				if (shiftKeyDown) {
					canvas.remove((DrawableToken) t);
					addToCanvasTop(new DrawableText(t.get_label(), me.getX(), me.getY(), true, t.get_colour()));
				}
				else {
					canvas.getInterpreter().setAutoConstraintSolving(false);
					t.set_mid_x(me.getX());
					canvas.getInterpreter().setAutoConstraintSolving(true);
					t.set_mid_y(me.getY());
				}
				updateStatusField();
				canvas.repaint();
			}
		}
		if (current instanceof Circle) {
			Circle c = (Circle) current;
			if (startX > ((int) c.get_mid_x() - MOUSE_TOLERANCE) && startX < ((int) c.get_mid_x() + MOUSE_TOLERANCE) &&
			    startY > ((int) c.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) c.get_mid_y() + MOUSE_TOLERANCE)) {
				if (shiftKeyDown) {
					canvas.remove((DrawableToken) c);
					addToCanvasTop(new DrawableCircle(c.get_radius(), me.getX(), me.getY(), true, c.get_colour()));
				}
				else {
					canvas.getInterpreter().setAutoConstraintSolving(false);
					c.set_mid_x(me.getX());
					canvas.getInterpreter().setAutoConstraintSolving(true);
					c.set_mid_y(me.getY());
				}
				updateStatusField();
				canvas.repaint();
			}
			else if (startX > ((int) (c.get_mid_x() + c.get_radius()) - MOUSE_TOLERANCE) && 
			    startX < ((int) (c.get_mid_x() + c.get_radius()) + MOUSE_TOLERANCE) &&
			    startY > ((int) c.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) c.get_mid_y() + MOUSE_TOLERANCE) &&
			    me.getX() > (int) (c.get_mid_x()) + MIN_RADIUS) {
				if (shiftKeyDown) {
					canvas.remove((DrawableToken) c);
					addToCanvasTop(new DrawableCircle(me.getX() - c.get_mid_x(), c.get_mid_x(), c.get_mid_y(), true, 
						c.get_colour()));
				}
				else {
					c.set_radius(me.getX() - c.get_mid_x());
				}
				updateStatusField();
				canvas.repaint();
			}
		}
		if (current instanceof Arrow) {
			Arrow a = (Arrow) current;
			if (startX > ((int) a.get_start_x() - MOUSE_TOLERANCE) && startX < ((int) a.get_start_x() + MOUSE_TOLERANCE) &&
			    startY > ((int) a.get_start_y() - MOUSE_TOLERANCE) && startY < ((int) a.get_start_y() + MOUSE_TOLERANCE) &&
			    !(me.getX() == (int) a.get_end_x() && me.getY() == (int) a.get_end_y())) {
				if (shiftKeyDown) {
					canvas.remove((DrawableToken) a);
					addToCanvasTop(new DrawableArrow(me.getX(), me.getY(), 
						a.get_mid_x(), a.get_mid_y(), a.get_end_x(), a.get_end_y(), 0.0, 0.0, true, a.get_colour()));
				}
				else {
					canvas.getInterpreter().setAutoConstraintSolving(false);
					a.set_start_x(me.getX());
					canvas.getInterpreter().setAutoConstraintSolving(true);
					a.set_start_y(me.getY());
				}
				updateStatusField();
				canvas.repaint();
			}
			else if (startX > ((int) a.get_end_x() - MOUSE_TOLERANCE) && startX < ((int) a.get_end_x() + MOUSE_TOLERANCE) &&
			    startY > ((int) a.get_end_y() - MOUSE_TOLERANCE) && startY < ((int) a.get_end_y() + MOUSE_TOLERANCE) &&
			    !(me.getX() == (int) a.get_start_x() && me.getY() == (int) a.get_start_y())) {
				if (shiftKeyDown) {
					canvas.remove((DrawableToken) a);
					addToCanvasTop(new DrawableArrow(a.get_start_x(), a.get_start_y(), 
						a.get_mid_x(), a.get_mid_y(), me.getX(), me.getY(), 0.0, 0.0, true, a.get_colour()));
				}
				else {
					canvas.getInterpreter().setAutoConstraintSolving(false);
					a.set_end_x(me.getX());
					canvas.getInterpreter().setAutoConstraintSolving(true);
					a.set_end_y(me.getY());
				}
				updateStatusField();
				canvas.repaint();
			}
			else if (startX > ((int) a.get_mid_x() - MOUSE_TOLERANCE) && startX < ((int) a.get_mid_x() + MOUSE_TOLERANCE) &&
			    startY > ((int) a.get_mid_y() - MOUSE_TOLERANCE) && startY < ((int) a.get_mid_y() + MOUSE_TOLERANCE)) {
				if (shiftKeyDown) {
					canvas.remove((DrawableToken) a);
					addToCanvasTop(new DrawableArrow(a.get_start_x(), a.get_start_y(), 
						me.getX(), me.getY(), a.get_end_x(), a.get_end_y(), 0.0, 0.0, true, a.get_colour()));
				}
				else {
					canvas.getInterpreter().setAutoConstraintSolving(false);
					a.set_mid_x(me.getX());
					canvas.getInterpreter().setAutoConstraintSolving(true);
					a.set_mid_y(me.getY());
				}
				updateStatusField();
				canvas.repaint();
			}
		}
		mouseDrag = false;
	}

	private void addToCanvasTop(DrawableToken dt)
	{
		canvas.add(dt);
		canvas.moveToTop(dt);
	}

	// Tag all states and transitions according to which FSA they belong to
	private void tagFSAs()
	{
		Set stateSet = new HashSet();
		Set transitionSet = new HashSet();

		// Collect all states and transitions and set their tags to zero
		Iterator iter = canvas.getInterpreter().getAllSymbols().iterator();
		while (iter.hasNext()) {
			Object obj = iter.next();
			if (obj instanceof State) {
				((State) obj).set_tag(0);
				stateSet.add(obj);
			}
			else if (obj instanceof Transition) {
				((Transition) obj).set_tag(0);
				transitionSet.add(obj);
			}
		}

		// Tag all states
		int id = 1;
		iter = stateSet.iterator();
		while (iter.hasNext()) {
			State state = (State) iter.next();
			tagState(state, id, transitionSet);
			id++;
		}
	}

	// Tags a state and any transitions involving the state
	private void tagState(State state, int id, Set transitionSet)
	{
		// Ignore if already tagged
		if (state.get_tag() != 0)
			return;

		// Tag state
		state.set_tag(id);

		// Go through all transitions
		Iterator iter = transitionSet.iterator();
		while (iter.hasNext()) {
			Transition transition = (Transition) iter.next();
			if (transition.get_tag() == 0) {
				transition.set_tag(id);

				if (transition.get_from() == state)
					tagState(transition.get_to(), id, transitionSet);
				else if (transition.get_to() == state)
					tagState(transition.get_from(), id, transitionSet);
			}
		}
	}

	// Implement grammar symbol listener methods
	public void symbolModified(GrammarSymbolEvent e) { ; }

	public void symbolAdded(GrammarSymbolEvent e)
	{
		System.out.println("" + e.getGrammarSymbol().getName() + " symbol added to the parse forest");
	}

	public void symbolRemoved(GrammarSymbolEvent e)
	{
		System.out.println("" + e.getGrammarSymbol().getName() + " symbol removed from the parse forest");
	}

	// Main method
	public static void main(String argv[])
	{
		JFrame editor = new Editor();
	}
}

